#!/usr/bin/env python3
# -*- coding:utf-8 -*-
try:
    import sys
    import importlib

    importlib.reload(sys)
    import os
    import subprocess
    import configparser
    from gspylib.inspection.common.CheckItem import BaseItem
    from gspylib.inspection.common.CheckResult import ResultStatus
    from gspylib.os.gsfile import g_file
    from gspylib.os.gsplatform import Platform
    from gspylib.hardware.gsmemory import g_memory
except ImportError as ie:
    raise Exception("[GAUSS-52200] : Unable to import module: %s." % str(ie))

setParameterList = {}


class CheckSysParams(BaseItem):
    def __init__(self):
        super(CheckSysParams, self).__init__(self.__class__.__name__)
        self.version = None

    def preCheck(self):
        # check the threshold was set correctly
        if (not self.threshold.__contains__('version')):
            raise Exception("version can not be empty")
        self.version = self.threshold['version']

    def SetSysctlForList(self, key, value):
        """
        function: Set sysctl parameter
        input : key, value
        output: NA
        """
        kernelParameterFile = "/etc/sysctl.conf"
        cmd = """sed -i '/^\\s*%s *=.*$/d' %s &&
               echo %s = %s  >> %s 2>/dev/null""" % (key, kernelParameterFile, key, value, kernelParameterFile)
        (status, output) = subprocess.getstatusoutput(cmd)
        if (status != 0):
            raise Exception(output)

    def getConfigFilePara(self, configFile, section, checkList=None, optionsName=None):
        if checkList is None:
            checkList = []
        if optionsName is None:
            optionsName = []
        try:
            data = {}
            fp = configparser.RawConfigParser()
            fp.read(configFile)
            secs = fp.sections()
            if section not in secs:
                return data
            optionList = fp.options(section)
            if (len(optionsName) != 0 and optionsName not in optionList):
                return data
            elif (len(optionsName) != 0):
                optionList = optionsName
            for key in optionList:
                value = fp.get(section, key)
                if (len(value.split()) == 0):
                    raise Exception("The parameter value of %s is Null." % key)
                value = value.split('#')[0]
                if (key in checkList and not value.isdigit()):
                    raise Exception("The value of %s must be a digit." % key)
                if (section == '/etc/security/limits.conf' and not value.isdigit() and value != 'unlimited'):
                    raise Exception("The value of %s is error." % key)
                data[key] = value
            if ("vm.min_free_kbytes" in data.keys()):
                output = g_memory.getMemTotalSize()
                totalMemory_k = output // 1024
                multiple = data["vm.min_free_kbytes"].split('*')[1].split('%')[0].strip()
                val = int(totalMemory_k) * int(multiple) // 100
                data["vm.min_free_kbytes"] = str(val)

            if "net.nf_conntrack_max" in data.keys() and \
                    not os.path.exists('/proc/sys/net/nf_conntrack_max'):
                data.pop('net.nf_conntrack_max')

            return data
        except Exception as e:
            raise Exception(
                "The configuration file [%s] contains parsing errors." % configFile + " Error: \n%s" % str(e))

    def getpatchlevel(self):
        gs_platform = Platform()
        distname, version, _ = gs_platform.dist()
        if (distname == "SuSE" and version == "11"):
            patInfo = g_file.readFile("/etc/SuSE-release", 'PATCHLEVEL')[0]
            if (patInfo.find('=') > 0):
                output = patInfo.split('=')[1].strip()
                if (output != "" and output == "1"):
                    return True
        return False

    def doCheck(self):
        global setParameterList
        resultList = []
        patchlevel = self.getpatchlevel()
        informationlist = []
        dirName = os.path.dirname(os.path.realpath(__file__))

        configFile = "%s/../../config/check_list_%s.conf" % (dirName, self.version)
        suggestParameterList = self.getConfigFilePara(configFile, 'SUGGEST:/etc/sysctl.conf')
        kernelParameter = self.getConfigFilePara(configFile, '/etc/sysctl.conf')
        kernelParameter.update(suggestParameterList)
        for key in kernelParameter:
            if (patchlevel and key == "vm.extfrag_threshold"):
                continue
            if (key == "sctpchecksumerrors"):
                snmpFile = "/proc/net/sctp/snmp"
                if (os.path.isfile(snmpFile)):
                    output = g_file.readFile(snmpFile, 'SctpChecksumErrors')[0].split()[1].strip()
                else:
                    continue
            else:
                sysFile = "/proc/sys/%s" % key.replace('.', '/')
                # High version of linux no longer supports tcp_tw_recycle
                if (not os.path.exists(sysFile) and key == "net.ipv4.tcp_tw_recycle"):
                    continue
                output = g_file.readFile(sysFile)[0].strip()
            if (len(output.split()) > 1):
                output = ' '.join(output.split())

            if (output != kernelParameter[key].strip() and key not in suggestParameterList.keys()):
                resultList.append(1)
                informationlist.append("Abnormal reason: variable '%s' RealValue '%s' ExpectedValue '%s'." % (
                    key, output, kernelParameter[key]))
                setParameterList[key] = kernelParameter[key]
            elif output != kernelParameter[key].strip():
                if (key == "vm.overcommit_ratio"):
                    output = g_file.readFile("/proc/sys/vm/overcommit_memory")[0].strip()
                    if (output == "0"):
                        continue
                resultList.append(2)
                informationlist.append("Warning reason: variable '%s' RealValue '%s' ExpectedValue '%s'." % (
                    key, output, kernelParameter[key]))
        if (1 in resultList):
            self.result.rst = ResultStatus.NG
        elif (2 in resultList):
            self.result.rst = ResultStatus.WARNING
        else:
            self.result.rst = ResultStatus.OK
        self.result.val = "\n".join(informationlist)

    def delSysctlForList(self, key, value):
        """
        """
        kernelParameterFile = "/etc/sysctl.conf"
        cmd = """sed -i '/^\\s*%s *=.*$/d' %s """ % (key, kernelParameterFile)
        (status, _) = subprocess.getstatusoutput(cmd)
        if status != 0:
            raise Exception("Failed to delete variable '%s %s' from /etc/sysctl.conf." % (key, value))

    def doSet(self):
        for checkResult in self.result.val.split('\n'):
            if checkResult.startswith("Abnormal reason"):
                checkResultList = checkResult.split('\'')
                setParameterList[checkResultList[1]] = checkResultList[5]
        self.result.val = ""
        # The parameter sctpchecksumerrors set method is independent
        if "sctpchecksumerrors" in setParameterList:
            cmd = "echo 1 > /sys/module/sctp/parameters/no_checksums"
            (status, output) = subprocess.getstatusoutput(cmd)
            if (status != 0):
                self.result.val += "Failed to enforce sysctl kernel variable 'sctpchecksumerrors'. Error: %s" % output
            setParameterList.pop("sctpchecksumerrors")

        if len(setParameterList) != 0:
            for key in setParameterList:
                self.SetSysctlForList(key, setParameterList[key])
                self.result.val = "%sSet variable '%s' to '%s'\n" % (self.result.val, key, setParameterList.get((key)))
            cmd = "sysctl -p"
            (status, output) = subprocess.getstatusoutput(cmd)
            if status != 0:
                cmderrorinfo = "sysctl -p | grep 'No such file or directory'"
                (status, outputresult) = subprocess.getstatusoutput(cmderrorinfo)
                if status != 0 and outputresult == "":
                    raise Exception(output)
                for key in setParameterList:
                    tmp = "/proc/sys/%s" % key.replace('.', '/')
                    if tmp in outputresult or key in outputresult:
                        # delete the record about key from the /etc/sysctl.conf
                        self.delSysctlForList(key, setParameterList[key])
