#!/usr/bin/env python3
# -*- coding:utf-8 -*-
try:
    import sys
    import importlib

    importlib.reload(sys)
    import platform
    from gspylib.inspection.common.CheckItem import BaseItem
    from gspylib.inspection.common.CheckResult import ResultStatus
    from gspylib.os.platform import support_platform
    from gspylib.os.gsplatform import g_Platform
except Exception as ie:
    raise Exception("[GAUSS-52200] : Unable to import module: %s." % str(ie))


class CheckOSVer(BaseItem):
    def __init__(self):
        super(CheckOSVer, self).__init__(self.__class__.__name__)

    def doCheck(self):
        (distName, version) = g_Platform.getCurrentPlatForm()
        bits, _ = platform.architecture()
        distName = distName.lower()
        self.result.val = "The current OS is %s %s %s" % (distName, version, bits)
        if distName in support_platform.RHEL_SERIES_VERSION_LIST + support_platform.OTHER_PLATFORM_LIST:
            self.result.rst = ResultStatus.OK
            self.result.val = "The current OS is %s %s %s." % (distName, version[0:3], bits)
        elif distName == support_platform.SUSE:
            self.result.rst = ResultStatus.OK
            self.result.val = "The current OS is SuSE %s 64bit." % version
        else:
            self.result.rst = ResultStatus.NG
            self.result.val = "The current OS[%s %s] does not meet the requirements." % (distName, version)

    def postAnalysis(self, itemResult, category="", name=""):
        errors = []
        itemResult.standard = self.standard
        for i in itemResult.getLocalItems():
            if i.rst == ResultStatus.NG:
                errors.append("%s: %s" % (i.host, i.val))
        if len(errors) > 0:
            itemResult.rst = ResultStatus.NG
            itemResult.analysis = "\n".join(errors)
            return itemResult

        analysis = ""
        VerGroupDisk = {'RedHat6': [], 'RedHat7': [], 'Euler': [], 'SuSE11SP1': [], 'SuSE11SP234': [], 'SuSE12': []}
        for v in itemResult.getLocalItems():
            analysis = "%s%s: %s\n" % (analysis, v.host, v.val)
            verInfo = v.val.strip().split(' ')[4:]
            if verInfo[0] in support_platform.RHEL_MASTER_PLATFORM_LIST:
                if verInfo[1][0] == support_platform.RHEL6:
                    VerGroupDisk['RedHat6'].append(verInfo)
                else:
                    VerGroupDisk['RedHat7'].append(verInfo)
            elif verInfo[0] in support_platform.OTHER_PLATFORM_LIST + [support_platform.EULEROS]:
                VerGroupDisk['Euler'].append(verInfo)
            elif verInfo[0] == support_platform.SUSE:
                version = verInfo[1].split('.')[0]
                if version == support_platform.SUSE11:
                    if verInfo[1] == "11.1":
                        VerGroupDisk['SuSE11SP1'].append(verInfo)
                    else:
                        VerGroupDisk['SuSE11SP234'].append(verInfo)
                else:
                    VerGroupDisk['SuSE12'].append(verInfo)
        currentVerGroup = []
        for verGroup in VerGroupDisk.keys():
            if len(VerGroupDisk[verGroup]) != 0:
                currentVerGroup.append(verGroup)
        if len(currentVerGroup) > 1:
            itemResult.rst = ResultStatus.NG
        else:
            itemResult.rst = ResultStatus.OK
        itemResult.analysis = analysis

        return itemResult
